﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using PpmsDataWebService.Models;
using VA.PPMS.Context;

namespace PpmsDataWebService.Mappers
{
    public class ProviderServicesMap
    {
        public static async Task<IQueryable<ProviderService>> MapProviderServices(List<ppms_providerservice> ppmsProviderServices, string mapRelatedType, PpmsContext context)
        {
            var providerServices = new List<ProviderService> {};
            foreach (var ppmsProviderService in ppmsProviderServices)
            {

                //Populate the Provider Service Details
                var providerService = new ProviderService
                {
                    DescriptionOfService = ppmsProviderService.ppms_descriptionofservice,
                    Limitation = ppmsProviderService.ppms_limitation
                };

                if(ppmsProviderService.ppms_HighPerformingProvider != null)
                {
                    switch (ppmsProviderService.ppms_HighPerformingProvider.Value)
                    {
                        case (int)ppms_providerservice_ppms_HighPerformingProvider.Unknown:
                            providerService.HPP = HPP.Unknown;
                            break;                          
                        case (int)ppms_providerservice_ppms_HighPerformingProvider.Yes:
                            providerService.HPP = HPP.Yes;
                            break;
                        case (int)ppms_providerservice_ppms_HighPerformingProvider.No:
                            providerService.HPP = HPP.No;
                            break;
                    }
                }

                
                if (ppmsProviderService.ppms_name != null)
                { providerService.Name = ppmsProviderService.ppms_name; }
                else { providerService.Name = " "; }

                //Lookup attribute names
                if (ppmsProviderService.ppms_network != null)
                {
                    providerService.AffiliationName = ppmsProviderService.ppms_network.Name;

                    if (ppmsProviderService.ppms_HighPerformingProvider != null)
                    {
                        switch (ppmsProviderService.ppms_HighPerformingProvider.Value)
                        {
                            case (int)ppms_providerservice_ppms_HighPerformingProvider.Unknown:
                                providerService.HighPerformingProvider = String.Format("{0}(U)", ppmsProviderService.ppms_network.Name);
                                break;
                            case (int)ppms_providerservice_ppms_HighPerformingProvider.Yes:
                                providerService.HighPerformingProvider = String.Format("{0}(Y)", ppmsProviderService.ppms_network.Name);
                                break;
                            case (int)ppms_providerservice_ppms_HighPerformingProvider.No:
                                providerService.HighPerformingProvider = String.Format("{0}(N)", ppmsProviderService.ppms_network.Name);
                                break;
                        }
                    }



                }
                if (ppmsProviderService.ppms_VAProviderRelationshipId != null)
                    providerService.RelationshipName = ppmsProviderService.ppms_VAProviderRelationshipId.Name;
                    
                if (ppmsProviderService.ppms_ProviderId != null)
                    providerService.ProviderName = ppmsProviderService.ppms_ProviderId.Name;

                if (ppmsProviderService.ppms_ProviderAgreement != null)
                    providerService.ProviderAgreementName = ppmsProviderService.ppms_ProviderAgreement.Name;

                if (ppmsProviderService.ppms_specialty != null)
                    providerService.SpecialtyName = ppmsProviderService.ppms_specialty.Name;


                if (ppmsProviderService.ppms_caresite != null)
                {                   
                    var careSite = context.ppms_caresiteSet.First(i => i.Id == ppmsProviderService.ppms_caresite.Id);
                    providerService.CareSitePhoneNumber = careSite.ppms_mainsitephone;
                    providerService.CareSiteLocationAddress = careSite.ppms_address_compositeid;
                    providerService.CareSiteName = careSite.ppms_name;
                    providerService.CareSiteAddressStreet = careSite.ppms_address_line1;
                    providerService.CareSiteAddressCity = careSite.ppms_address_city;
                    providerService.CareSiteAddressState = careSite.ppms_statename;
                    providerService.CareSiteAddressZipCode = careSite.ppms_address_postalcode;
                }

                if (ppmsProviderService.ppms_latitude != null)
                {
                    providerService.Latitude = (decimal)ppmsProviderService.ppms_latitude;
                }
                if (ppmsProviderService.ppms_longitude != null)
                {
                    providerService.Longitude = (decimal)ppmsProviderService.ppms_longitude;
                }

                if (ppmsProviderService.ppms_Organization != null)
                    providerService.OrganiztionGroupName = ppmsProviderService.ppms_Organization.Name;

                //Extract the Related type from queryType string
                if (mapRelatedType.Contains("?$expand=Specialty"))
                    mapRelatedType = "Specialty";
                if (mapRelatedType.Contains("?$expand=Provider"))
                    mapRelatedType = "Provider";
                if (mapRelatedType.Contains("?$expand=ServiceAvailabilities"))
                    mapRelatedType = "ServiceAvailabilities";
                if (mapRelatedType.Contains("?$expand=AgreementProviders"))
                    mapRelatedType = "AgreementProviders";

                //Map Related types switch. 
                switch (mapRelatedType)
                {                    
                    case "Specialty":
                        //Lookup and Map the Specialty
                        var ppmsSpecialty =
                            context.ppms_taxonomySet.FirstOrDefault(i => i.Id == ppmsProviderService.ppms_specialty.Id);
                        var ppmsSpecialtiesList = new List<ppms_taxonomy> { ppmsSpecialty };
                        providerService.Specialty =
                            await SpecialtyMap.MapSpecialties(ppmsSpecialtiesList, "None", context);
                        break;
                    case "Provider":
                        //Lookup and Map the Provider
                        var ppmsProvider =
                            context.AccountSet.FirstOrDefault(i => i.Id == ppmsProviderService.ppms_ProviderId.Id);
                        var ppmsProvidersList = new List<Account> { ppmsProvider };
                        providerService.Provider =
                            await ProviderMap.MapProviders(ppmsProvidersList, "None");
                        break;
                    case "ServiceAvailabilities":
                        var ppmsServiceAvailabilities =
                            context.ppms_serviceavailabilitySet.Where(i => i.ppms_ProviderServiceId.Id == ppmsProviderService.Id);
                        var ppmsServiceAvailabilitiesList = ppmsServiceAvailabilities.ToList();
                        providerService.ServiceAvailabilities =
                            await ServiceAvailabilityMap.MapServiceAvailabilities(ppmsServiceAvailabilitiesList, "None",
                                context);
                        break;
                    case "AgreementProviders":
                        var ppmsAgreementProviders =
                            context.ppms_agreementproviderSet.Where(
                                i => i.ppms_ProviderAgreement.Id == ppmsProviderService.Id);
                        var ppmsAgreementProvidersList = ppmsAgreementProviders.ToList();
                        providerService.AgreementProviders =
                            await AgreementProvidersMap.MapAgreementProviders(ppmsAgreementProvidersList, "None",
                                context);
                        break;
                    default:
                        break;
                }



                providerServices.Add(providerService);
                
            }
            return providerServices.AsQueryable();            
        }

        
    }
}